<?php

require_once DIR_SYSTEM . 'library/vendor/sezzle/utils/platform.php';

use Sezzle\HttpClient\ClientService;
use Sezzle\HttpClient\Exception\InvalidRequest;
use Sezzle\Model\AuthCredentials;
use Sezzle\Model\Order\Capture;
use Sezzle\Model\Order\Refund;
use Sezzle\Model\Order\Release;
use Sezzle\Services\AuthenticationService;
use Sezzle\Platform;


/**
 * Class ModelExtensionPaymentSezzle
 */
class ModelExtensionPaymentSezzle extends Model
{
    const MODE_SANDBOX = "sandbox";
    const MODE_PRODUCTION = "production";
    const MODE_AUTH = "authorize";
    const MODE_AUTHCAPTURE = "authorize_capture";
    const PAYMENT_ACTION_CAPTURE = "capture";
    const PAYMENT_ACTION_REFUND = "refund";
    const PAYMENT_ACTION_RELEASE = "release";
    const ORDER_STATE_APPROVED = "Approved";
    const ORDER_STATE_REFUNDED = "Refunded";
    const ORDER_STATE_RELEASED = "Released";

    /****************************
     * SEZZLE SERVICE FUNCTIONS *
     ****************************/

    /**
     * Get auth headers
     *
     * @return string[]
     * @throws InvalidRequest
     */
    private function getAuthHeaders()
    {
        return $this->getHeaders($this->getToken());
    }

    /**
     * Get headers
     *
     * @param string|null $token
     * @return string[]
     */
    private function getHeaders($token = '')
    {
        if ($token) {
            return ['Authorization: Bearer ' . $token];
        }

        $encodedData = base64_encode(json_encode(Platform::getPlatformData()));
        return ['Sezzle-Platform: ' . $encodedData];
    }

    /**
     * Get Authentication Token
     *
     * @return string
     * @throws InvalidRequest
     */
    public function getToken($request = [])
    {
        $api_mode = ($request['api_mode']
            ?? $this->config->get('payment_sezzle_test_mode')) ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $public_key = ($request['public_key'] ?? $this->config->get('payment_sezzle_public_key'));
        $private_key = ($request['private_key'] ?? $this->config->get('payment_sezzle_private_key'));
        $gateway_region = ($request['gateway_region'] ?? $this->config->get('payment_sezzle_gateway_region'));

        // auth credentials set
        $auth_model = new AuthCredentials();
        $auth_model->setPublicKey($public_key)->setPrivateKey($private_key);

        // instantiate authentication service
        $token_service = new AuthenticationService(new ClientService(
            $api_mode,
            $gateway_region,
            $this->getHeaders()
        ));

        $token_model = $token_service->get($auth_model->toArray());
        return $token_model->getToken();
    }

    /**
     * Capturing the payment
     *
     * @param float $amount
     * @param string $currency_code
     * @param string $order_uuid
     * @param bool $is_partial_capture
     * @return Capture
     * @throws InvalidRequest
     */
    public function capturePayment($amount, $currency_code, $order_uuid, $is_partial_capture)
    {
        $api_mode = $this->config->get('payment_sezzle_test_mode') ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');
        $payload = $this->buildCapturePayload($amount, $currency_code, $is_partial_capture);

        // instantiate capture service
        $capture_service = new Sezzle\Services\CaptureService(new ClientService(
            $api_mode,
            $gateway_region,
            $this->getAuthHeaders()
        ));

        // get capture response
        return $capture_service->capturePayment($order_uuid, $payload->toArray());
    }

    /**
     * Build Capture Payload
     *
     * @param float $amount
     * @param string $currency_code
     * @param bool $is_partial_capture
     * @return Capture
     */
    public function buildCapturePayload($amount, $currency_code, $is_partial_capture)
    {
        $capture_model = new Sezzle\Model\Order\Capture();
        return $capture_model->setCaptureAmount(
            $this->getAmountObject(
                Sezzle\Util::formatToCents($amount),
                $currency_code
            )
        )
            ->setPartialCapture($is_partial_capture);
    }

    /**
     * Get Amount Object
     *
     * @param int $amount_cents
     * @param string $currency_code
     * @return Sezzle\Model\Session\Order\Amount
     */
    private function getAmountObject($amount_cents, $currency_code)
    {
        // amount model
        $amount_model = new Sezzle\Model\Session\Order\Amount();
        return $amount_model->setAmountInCents($amount_cents)->setCurrency($currency_code);
    }

    /**
     * Refunding the payment
     *
     * @param string $order_uuid
     * @param float $amount
     * @param string $currency_code
     * @return Refund
     * @throws InvalidRequest
     */
    public function refundPayment($order_uuid, $amount, $currency_code)
    {
        $api_mode = $this->config->get('payment_sezzle_test_mode') ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');
        $payload = $this->buildRefundPayload($amount, $currency_code);

        // instantiate refund service
        $refund_service = new Sezzle\Services\RefundService(new ClientService(
            $api_mode,
            $gateway_region,
            $this->getAuthHeaders()
        ));

        // get refund response
        return $refund_service->refundPayment(
            $order_uuid,
            $payload->toArray()
        );
    }

    /**
     * Build Refund Payload
     *
     * @param float $amount
     * @param string $currencyCode
     * @return Sezzle\Model\Session\Order\Amount
     */
    private function buildRefundPayload($amount, $currencyCode)
    {
        return $this->getAmountObject(
            Sezzle\Util::formatToCents($amount),
            $currencyCode
        );
    }

    /**
     * Releasing the payment
     *
     * @param string $order_uuid
     * @param float $amount
     * @param string $currency_code
     * @return Release
     * @throws InvalidRequest
     */
    public function releasePayment($order_uuid, $amount, $currency_code)
    {
        $api_mode = $this->config->get('payment_sezzle_test_mode') ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');
        $payload = $this->buildReleasePayload($amount, $currency_code);

        // instantiate release service
        $release_service = new Sezzle\Services\ReleaseService(new ClientService(
            $api_mode,
            $gateway_region,
            $this->getAuthHeaders()
        ));

        // get release response
        return $release_service->releasePayment(
            $order_uuid,
            $payload->toArray()
        );
    }

    /**
     * Build Release Payload
     *
     * @param float $amount
     * @param string $currency_code
     * @return Sezzle\Model\Session\Order\Amount
     */
    private function buildReleasePayload($amount, $currency_code)
    {
        return $this->getAmountObject(
            Sezzle\Util::formatToCents($amount),
            $currency_code
        );
    }

    /**
     * Send config data to Sezzle
     *
     * @param array $config
     * @return bool
     * @throws InvalidRequest
     */
    public function sendConfig($config)
    {
        $api_mode = $config['payment_sezzle_test_mode'] ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $gateway_region = $config['payment_sezzle_gateway_region'];
        $payload = $this->buildConfigPayload($config);

        $headers = $this->getHeaders($this->getToken([
            'api_mode' => $api_mode,
            'public_key' => $config['payment_sezzle_public_key'],
            'private_key' => $config['payment_sezzle_private_key'],
            'gateway_region' => $gateway_region,
        ]));

        // instantiate config service
        $config_service = new Sezzle\Services\ConfigService(new ClientService(
            $api_mode,
            $gateway_region,
            $headers
        ));

        // get config response
        return $config_service->sendConfig($payload);
    }

    /**
     * Build config payload
     *
     * @param array $config_data
     * @return array
     */
    private function buildConfigPayload($config_data)
    {
        return [
            'sezzle_enabled' => (bool)$config_data['payment_sezzle_status'],
            'merchant_uuid' => $config_data['payment_sezzle_merchant_uuid'],
            'pdp_widget_enabled' => (bool)$config_data['payment_sezzle_widget_pdp'],
            'cart_widget_enabled' => (bool)$config_data['payment_sezzle_widget_cart'],
            'installment_widget_enabled' => true,
            'payment_action' => $config_data['payment_sezzle_transaction_method'],
            'tokenization_enabled' => (bool)$config_data['payment_sezzle_tokenization'],
            'store_url' => HTTPS_SERVER
        ];
    }

    /****************************
     * SEZZLE SERVICE FUNCTIONS *
     ****************************/

    /********************
     * HELPER FUNCTIONS *
     ********************/

    /**
     * Validating amount by payment action type
     *
     * @param array $txn
     * @param float $amount
     * @param string $type
     * @return bool
     */
    public function validateAmount($txn, $amount, $type = '')
    {
        if (!$type) {
            return false;
        }

        $amountAvailable = 0.00;
        switch ($type) {
            case self::PAYMENT_ACTION_CAPTURE:
                $amountAvailable = $txn['checkout_amount'] - $txn['captured_amount'];
                break;
            case self::PAYMENT_ACTION_REFUND:
                $amountAvailable = $txn['captured_amount'] - $txn['refunded_amount'];
                break;
            case self::PAYMENT_ACTION_RELEASE:
                $amountAvailable = $txn['authorized_amount'] - $txn['captured_amount'];
                break;
        }

        return $amount <= $amountAvailable;
    }

    /**
     * Getting order status id by order state
     *
     * @param string $state
     * @return bool
     */
    public function getOrderStatusId($state)
    {
        $this->load->model('localisation/order_status');

        $order_statuses = $this->model_localisation_order_status->getOrderStatuses();
        foreach ($order_statuses as $order_status) {
            switch ($state) {
                case self::ORDER_STATE_APPROVED:
                    if ($order_status['name'] === 'Processing') {
                        return $order_status['order_status_id'];
                    }
                    break;
                case self::ORDER_STATE_REFUNDED:
                    if ($order_status['name'] === 'Refunded') {
                        return $order_status['order_status_id'];
                    }
                    break;
                case self::ORDER_STATE_RELEASED:
                    if ($order_status['name'] === 'Voided') {
                        return $order_status['order_status_id'];
                    }
                    break;
            }
        }
        return false;
    }

    /********************
     * HELPER FUNCTIONS *
     ********************/

    /****************
     * DB FUNCTIONS *
     ****************/

    /**
     * SQL script for installing Sezzle
     */
    public function install()
    {
        $this->db->query("CREATE TABLE IF NOT EXISTS `" . DB_PREFIX . "sezzle_order` (
          `sezzle_order_id` int(11) NOT NULL AUTO_INCREMENT,
          `date_added` DATETIME NOT NULL,
          `date_modified` DATETIME NOT NULL,
          `opencart_order_id` int(11) NOT NULL,
          `order_uuid` varchar(255) NOT NULL,
          `checkout_url` varchar(255) NOT NULL,
          `checkout_amount` DECIMAL( 10, 2 ) NOT NULL,
          `authorized_amount` DECIMAL( 10, 2 ) NOT NULL,
          `captured_amount` DECIMAL( 10, 2 ) NOT NULL,
          `refunded_amount` DECIMAL( 10, 2 ) NOT NULL,
          `released_amount` DECIMAL( 10, 2 ) NOT NULL,
          `currency_code` CHAR(3) NOT NULL,
          `currency_value` DECIMAL( 15, 8 ) NOT NULL,
          `auth_expiration` DATETIME NULL,
          `transaction_method` varchar(255) NOT NULL,
          PRIMARY KEY (`sezzle_order_id`)
        ) ENGINE=InnoDB DEFAULT COLLATE=utf8_general_ci;");

        $this->db->query("CREATE TABLE IF NOT EXISTS `" . DB_PREFIX . "sezzle_order_transaction` (
          `sezzle_order_transaction_id` INT(11) NOT NULL AUTO_INCREMENT,
          `sezzle_order_id` INT(11) NOT NULL,
          `date_added` DATETIME NOT NULL,
          `type` ENUM('authorize', 'capture', 'refund', 'release') DEFAULT NULL,
          `amount` DECIMAL( 10, 2 ) NOT NULL,
          PRIMARY KEY (`sezzle_order_transaction_id`)
        ) ENGINE=InnoDB DEFAULT COLLATE=utf8_general_ci;");

        $this->db->query("CREATE TABLE IF NOT EXISTS `" . DB_PREFIX . "sezzle_tokenization` (
          `customer_id` int(11) NOT NULL AUTO_INCREMENT,
          `customer_uuid` varchar(255) NOT NULL,
          `customer_uuid_expiration` DATETIME NOT NULL,
          `approved` BOOLEAN NOT NULL,
          PRIMARY KEY (`customer_id`)
        ) ENGINE=InnoDB DEFAULT COLLATE=utf8_general_ci;");
    }

    /**
     * SQL script for uninstalling Sezzle
     */
    public function uninstall()
    {
        $this->db->query("DROP TABLE IF EXISTS `" . DB_PREFIX . "sezzle_order`");
        $this->db->query("DROP TABLE IF EXISTS `" . DB_PREFIX . "sezzle_order_transaction`");
        $this->db->query("DROP TABLE IF EXISTS `" . DB_PREFIX . "sezzle_tokenization`");
    }

    /**
     * Get Sezzle order by OC order ID
     *
     * @param int $order_id
     * @return bool
     */
    public function getOrder($order_id)
    {
        $query = $this->db->query(sprintf("SELECT * FROM %s WHERE opencart_order_id = %d LIMIT 1", DB_PREFIX . "sezzle_order", (int)$order_id));
        if (!$query->num_rows) {
            return false;
        }
        $query->row['transactions'] = $this->getTransactions($query->row['sezzle_order_id']);
        return $query->row;
    }

    /**
     * Add Sezzle order transaction
     *
     * @param int $sezzle_order_id
     * @param string $type
     * @param float $amount
     */
    public function addTransaction($sezzle_order_id, $amount, $type)
    {
        $this->db->query("INSERT INTO `" . DB_PREFIX . "sezzle_order_transaction` SET `sezzle_order_id` = " . (int)$sezzle_order_id . ", `date_added` = now(), `type` = '" . $this->db->escape($type) . "', `amount` = " . (float)$amount);
    }

    /**
     * Setting Released Amount
     *
     * @param int $order_id
     * @param float $amount
     */
    public function setReleasedAmount($order_id, $amount)
    {
        $this->db->query("UPDATE " . DB_PREFIX . "sezzle_order SET `released_amount` = " . (float)$amount . " WHERE opencart_order_id = " . (int)$order_id);
    }

    /**
     * Setting Captured Amount
     *
     * @param int $order_id
     * @param float $amount
     */
    public function setCapturedAmount($order_id, $amount)
    {
        $this->db->query("UPDATE " . DB_PREFIX . "sezzle_order SET `captured_amount` = " . (float)$amount . " WHERE opencart_order_id = " . (int)$order_id);
    }

    /**
     * Setting Refunded Amount
     *
     * @param int $order_id
     * @param float $amount
     */
    public function setRefundedAmount($order_id, $amount)
    {
        $this->db->query("UPDATE " . DB_PREFIX . "sezzle_order SET `refunded_amount` = " . (float)$amount . " WHERE opencart_order_id = " . (int)$order_id);
    }

    /**
     * Add order history
     *
     * @param int $order_id
     * @param int $order_status_id
     * @param string $comment
     */
    public function addOrderHistory($order_id, $order_status_id, $comment = '')
    {
        $this->db->query("UPDATE " . DB_PREFIX . "order SET `order_status_id` = " . (int)$order_status_id . " WHERE order_id = " . (int)$order_id);
        $this->db->query("INSERT INTO " . DB_PREFIX . "order_history SET order_id = '" . (int)$order_id . "', order_status_id = '" . (int)$order_status_id . "', notify = '0', comment = '" . $this->db->escape($comment) . "', date_added = NOW()");
    }

    /**
     * Get Sezzle order transaction
     *
     * @param int $sezzle_order_id
     * @return mixed
     */
    private function getTransactions($sezzle_order_id)
    {
        return $this->db->query(sprintf("SELECT * FROM %s WHERE sezzle_order_id = %d", DB_PREFIX . "sezzle_order_transaction", (int)$sezzle_order_id))->rows;
    }

    /****************
     * DB FUNCTIONS *
     ****************/
}
