<?php
require_once DIR_SYSTEM . 'library/vendor/sezzle/php-sdk/autoload.php';

use Sezzle\Config;

/**
 * Class ControllerExtensionPaymentSezzle
 */
class ControllerExtensionPaymentSezzle extends Controller
{

    /**
     * Configuration to Show Sezzle at Checkout
     */
    public function index()
    {
        $data = $this->load->language('extension/payment/sezzle');
        $this->load->model('checkout/order');
        $this->load->model('extension/payment/sezzle');

        $customer_id = isset($this->session->data['customer_id']) ? $this->session->data['customer_id'] : 0;
        $customer_uuid = $this->model_extension_payment_sezzle->getCustomerUUID($customer_id);
        $tokenization_enabled = $this->config->get('payment_sezzle_tokenization');
        $place_order_button = $this->language->get('button_continue_to_sezzle');
        $sezzle_banner = $this->language->get('text_sezzle_banner');
        if ($tokenization_enabled && $customer_uuid) {
            $place_order_button = $this->language->get('button_complete_order');
            $sezzle_banner = $this->language->get('text_sezzle_banner_tokenization');
        }

        $data = array_merge($data, [
            'button_continue' => $place_order_button,
            'text_loading' => $this->language->get('text_loading'),
            'testmode' => $this->config->get('payment_sezzle_test_mode'),
            'terms_href' => 'https://legal.sezzle.com/user',
        ]);

        // getting opencart order record
        $order_info = $this->model_checkout_order->getOrder($this->session->data['order_id']);
        if (empty($order_info)) {
            $data['continue'] = $this->url->link('checkout/cart');
            return $this->load->view('extension/payment/sezzle', $data);
        }

        // determining region text
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');
        $region = 'United States & Canada';
        if ($gateway_region === 'EU') {
            $region = 'Europe';
        }

        // template data
        $data = array_merge($data, [
            'sezzle_logo' => $this->language->get('text_sezzle_logo'),
            'sezzle_banner' => sprintf($sezzle_banner, $region),
            'sezzle_amount' => $this->currency->format($order_info['total'], $order_info['currency_code'], $order_info['currency_value'], false),
            'text_installment' => sprintf($this->language->get('text_installment'), $this->currency->format($order_info['total'] / 4, $order_info['currency_code'], $order_info['currency_value'], true)),
            'continue' => $this->url->link('extension/payment/sezzle/checkout', '', true)
        ]);

        return $this->load->view('extension/payment/sezzle', $data);
    }

    /**
     * Creating Checkout
     *
     * @throws Exception
     */
    public function checkout()
    {
        if ((!$this->cart->hasProducts() && empty($this->session->data['vouchers'])) || (!$this->cart->hasStock() && !$this->config->get('config_stock_checkout'))) {
            $this->response->redirect($this->url->link('checkout/cart'));
        }

        $this->load->model('extension/payment/sezzle');
        $this->load->model('checkout/order');
        $this->load->language('extension/payment/sezzle');

        $order_uuid = "";
        $checkout_url = "";
        $tokenization = false;

        $order_info = $this->model_checkout_order->getOrder($this->session->data['order_id']);
        $customer_id = isset($this->session->data['customer_id']) ? $this->session->data['customer_id'] : 0;

        try {
            $customer_uuid = $this->model_extension_payment_sezzle->getCustomerUUID($customer_id);
            $tokenization_enabled = $this->config->get('payment_sezzle_tokenization');
            if ($tokenization_enabled && $customer_uuid) { // tokenization order
                $order = $this->model_extension_payment_sezzle->createTokenizedOrder($customer_uuid, $order_info);
                if (!$order->getAuthorization()->isApproved()) {
                    throw new Exception($this->language->get('error_tokenize_order'));
                }
                $tokenization = true;
                $order_uuid = $order->getUuid();
            } else { // checkout session
                $session = $this->model_extension_payment_sezzle->createSession($order_info);
                if (!$session->getOrder()) {
                    $this->language->get('error_session')->$this->language->get('error_session');
                }
                $order_uuid = $session->getOrder()->getUuid();
                $checkout_url = $session->getOrder()->getCheckoutUrl();
            }
        } catch (Exception $e) {
            $this->session->data['error'] = $this->language->get('error_session');
            $this->response->redirect($this->url->link('checkout/checkout'));
        }

        $order_info['transaction_method'] = $this->config->get('payment_sezzle_transaction_method');
        $this->model_extension_payment_sezzle->addOrder($order_info, [
            "order_uuid" => $order_uuid,
            "checkout_url" => $checkout_url
        ]);
        $checkout_url = $tokenization ? $this->url->link('extension/payment/sezzle/checkoutComplete') : $session->getOrder()->getCheckoutUrl();
        $this->response->redirect($checkout_url);
    }

    /**
     * Completing Checkout
     *
     * @throws Exception
     */
    public function checkoutComplete()
    {
        $this->load->language('extension/payment/sezzle');
        $this->load->model('extension/payment/sezzle');
        $this->load->model('checkout/order');

        $order_id = $this->session->data['order_id'];

        // handling tokenization
        if (isset($this->session->data['customer_id']) && isset($this->request->get['customer-uuid'])) {
            $this->model_extension_payment_sezzle->handleTokenization($this->session->data['customer_id'], $this->request->get['customer-uuid']);
        }

        // getting opencart order record
        $order_info = $this->model_checkout_order->getOrder($order_id);
        if (empty($order_info)) {
            $this->session->data['error'] = $this->language->get('error_order_not_found');
            $this->response->redirect($this->url->link('checkout/checkout'));
        }

        // getting order record from sezzle_order table
        $txn = $this->model_extension_payment_sezzle->getOrder($order_id);
        $order_uuid = $txn['order_uuid'];
        $sezzle_order = $this->model_extension_payment_sezzle->getSezzleOrder($order_uuid);

        // validating amount
        $amount_matched = $this->model_extension_payment_sezzle->isAmountMatched($order_info, $sezzle_order);
        if (!$amount_matched) {
            $this->session->data['error'] = $this->language->get('error_amount_mismatch');
            $this->response->redirect($this->url->link('checkout/checkout'));
        }

        // validating auth
        $auth_valid = $this->model_extension_payment_sezzle->setAuthorization($sezzle_order, $order_info);
        if (!$auth_valid) {
            $this->session->data['error'] = $this->language->get('error_invalid_auth');
            $this->response->redirect($this->url->link('checkout/checkout'));
        }

        $this->model_extension_payment_sezzle->addTransaction($txn['sezzle_order_id'], $order_info['total'], 'authorize');

        $order_status_id = $this->config->get('config_order_status_id');

        // capturing payment
        $transaction_method = $this->config->get('payment_sezzle_transaction_method');
        if ($transaction_method === ModelExtensionPaymentSezzle::MODE_AUTHCAPTURE) {
            try {
                $this->model_extension_payment_sezzle->capturePayment($order_info, $order_uuid);
                $this->model_extension_payment_sezzle->setCapturedAmount($order_id, $txn['checkout_amount']);
                $this->model_extension_payment_sezzle->addTransaction($txn['sezzle_order_id'], $order_info['total'], 'capture');
                $order_status_id = $this->model_extension_payment_sezzle->getOrderStatusId(ModelExtensionPaymentSezzle::ORDER_STATE_CAPTURED);
            } catch (Exception $e) {
                $this->session->data['error'] = $this->language->get('error_capture');
                $this->response->redirect($this->url->link('checkout/checkout'));
            }
        }

        if (!$order_status_id) {
            $this->session->data['error'] = $this->language->get('error_checkout');
            $this->response->redirect($this->url->link('checkout/checkout'));
        }

        $this->model_checkout_order->addOrderHistory($order_id, $order_status_id);
        $this->response->redirect($this->url->link('checkout/success'));
    }

    /**
     * Event for rendering widget in PDP
     *
     * @param string $route
     * @param array $data
     * @param string $output
     */
    public function eventPostViewProductProduct($route, &$data, &$output)
    {
        $widget_config = $this->getWidgetConfig();
        if (!$widget_config['pdp_enabled']) {
            return;
        }

        $html = "<script type='text/javascript'>
                    document.sezzleConfig = {
						'configGroups': [
							{
								'targetXPath': '.list-unstyled/LI-0/H2-0',
								'renderToPath': '..',
								'fontSize': 12,
								'urlMatch': 'route=product/product'
							}
						]
					}
                </script>";
        $html .= "<script src='" . $widget_config['widget_url'] . "' type='text/javascript'></script>";
        $output = str_replace('</head>', $html . '</head>', $output);
    }

    /**
     * Event for rendering widget in Cart Page
     *
     * @param string $route
     * @param array $data
     * @param string $output
     */
    public function eventPostViewCheckoutCart($route, &$data, &$output)
    {
        $widget_config = $this->getWidgetConfig();
        if (!$widget_config['cart_enabled']) {
            return;
        }

        $html = "<script type='text/javascript'>
                    document.sezzleConfig = {
						'configGroups': [
							{
								'targetXPath': '.col-sm-4/.table/TBODY-0/TR-2/TD-1',
								'renderToPath': '../../..',
								'fontSize': 12,
								'urlMatch': 'route=checkout/cart'
							}
						]
					}
                </script>";
        $html .= "<script src='" . $widget_config['widget_url'] . "' type='text/javascript'></script>";
        $output = str_replace('</head>', $html . '</head>', $output);
    }

    /**
     * Get Widget Config
     *
     * @return array
     */
    private function getWidgetConfig()
    {
        $sezzle_enabled = $this->config->get('payment_sezzle_status');
        $pdp_widget_enabled = $this->config->get('payment_sezzle_widget_pdp');
        $cart_widget_enabled = $this->config->get('payment_sezzle_widget_cart');
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');
        $merchant_uuid = $this->config->get('payment_sezzle_merchant_uuid');
        $sezzle_domain = Config::getSezzleDomain($gateway_region);
        $widget_url = sprintf(
            'https://widget.%s/v1/javascript/price-widget?uuid=%s',
            $sezzle_domain,
            $merchant_uuid
        );

        $can_widget_be_enabled = $sezzle_enabled && $gateway_region && $merchant_uuid;
        return [
            'pdp_enabled' => $can_widget_be_enabled && $pdp_widget_enabled,
            'cart_enabled' => $can_widget_be_enabled && $cart_widget_enabled,
            'widget_url' => $widget_url,
        ];
    }
}
