<?php
require_once DIR_SYSTEM . 'library/vendor/sezzle/php-sdk/autoload.php';
require_once DIR_SYSTEM . 'library/vendor/sezzle/utils/platform.php';

use Sezzle\Config;
use Sezzle\HttpClient\ClientService;
use Sezzle\HttpClient\Exception\InvalidRequest;
use Sezzle\Model\AuthCredentials;
use Sezzle\Model\CustomerOrder;
use Sezzle\Model\Order;
use Sezzle\Model\Order\Capture;
use Sezzle\Model\Session;
use Sezzle\Services\AuthenticationService;
use Sezzle\Services\CustomerService;
use Sezzle\Services\OrderService;
use Sezzle\Services\SessionService;
use Sezzle\Services\TokenizationService;
use Sezzle\Util;
use Sezzle\Platform;

/**
 * Class ModelExtensionPaymentSezzle
 */
class ModelExtensionPaymentSezzle extends Model
{
    const REDIRECT_COMPLETE = "complete";
    const REDIRECT_CANCEL = "cancel";
    const MODE_SANDBOX = "sandbox";
    const MODE_PRODUCTION = "production";
    const MODE_AUTH = "authorize";
    const MODE_AUTHCAPTURE = "authorize_capture";
    const ORDER_STATE_CAPTURED = "CAPTURED";
    const ORDER_STATE_REFUNDED = "REFUNDED";
    const ORDER_STATE_RELEASED = "RELEASED";

    /**
     * Get payment method data
     *
     * @param array $address
     * @param float $total
     * @return array
     */
    public function getMethod($address, $total)
    {
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');
        if (!$gateway_region) {
            return [];
        }

        $this->load->language('extension/payment/sezzle');

        return [
            'code' => 'sezzle',
            'title' => $this->language->get('text_title'),
            'terms' => '',
            'sort_order' => $this->config->get('payment_sezzle_sort_order')
        ];
    }

    /****************************
     * SEZZLE SERVICE FUNCTIONS *
     ****************************/

    /**
     * Get auth headers
     *
     * @return string[]
     * @throws InvalidRequest
     */
    private function getAuthHeaders()
    {
        return $this->getHeaders($this->getToken());
    }

    /**
     * Get headers
     *
     * @param string|null $token
     * @return string[]
     */
    private function getHeaders($token = '')
    {
        if ($token) {
            return ['Authorization: Bearer ' . $token];
        }

        $encodedData = base64_encode(json_encode(Platform::getPlatformData()));
        return ['Sezzle-Platform: ' . $encodedData];
    }

    /**
     * Get Authentication Token
     *
     * @return string
     * @throws InvalidRequest
     */
    public function getToken()
    {
        $api_mode = $this->config->get('payment_sezzle_test_mode') ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $public_key = $this->config->get('payment_sezzle_public_key');
        $private_key = $this->config->get('payment_sezzle_private_key');
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');

        // auth credentials set
        $auth_model = new AuthCredentials();
        $auth_model->setPublicKey($public_key)->setPrivateKey($private_key);

        // instantiate authentication service
        $token_service = new AuthenticationService(new ClientService(
            $api_mode,
            $gateway_region,
            $this->getHeaders()
        ));

        $token_model = $token_service->get($auth_model->toArray());

        return $token_model->getToken();
    }

    /**
     * Creating Session
     *
     * @param array $order_info
     * @return Session
     * @throws InvalidRequest
     */
    public function createSession($order_info)
    {
        $payload = $this->buildSessionPayload($order_info);
        $api_mode = $this->config->get('payment_sezzle_test_mode') ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');

        // instantiate session service
        $session_service = new SessionService(new ClientService(
            $api_mode,
            $gateway_region,
            $this->getAuthHeaders()
        ));

        // session response
        return $session_service->createSession($payload->toArray());
    }

    /**
     * Capturing the Payment
     *
     * @param array $order_info
     * @param string $order_uuid
     * @return Capture
     * @throws InvalidRequest
     */
    public function capturePayment($order_info, $order_uuid)
    {
        $api_mode = $this->config->get('payment_sezzle_test_mode') ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');
        $payload = $this->buildCapturePayload($order_info);

        // instantiate capture service
        $capture_service = new Sezzle\Services\CaptureService(new ClientService(
            $api_mode,
            $gateway_region,
            $this->getAuthHeaders()
        ));

        // get capture response
        return $capture_service->capturePayment($order_uuid, $payload->toArray());
    }

    /**
     * Getting Order Information from Sezzle
     *
     * @param string $order_uuid
     * @return Order
     * @throws InvalidRequest
     */
    public function getSezzleOrder($order_uuid)
    {
        $api_mode = $this->config->get('payment_sezzle_test_mode') ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');

        // instantiate order service
        $order_service = new OrderService(new ClientService(
            $api_mode,
            $gateway_region,
            $this->getAuthHeaders()
        ));

        // order response
        return $order_service->getOrder($order_uuid);
    }

    /**
     * Create Tokenized Order
     *
     * @param array $order_info
     * @return CustomerOrder
     * @throws InvalidRequest
     * @throws Exception
     */
    public function createTokenizedOrder($customer_uuid, $order_info)
    {
        $payload = $this->buildCustomerOrderPayload($order_info);
        $api_mode = $this->config->get('payment_sezzle_test_mode') ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');

        // instantiate tokenization service
        $tokenization_service = new TokenizationService(new ClientService(
            $api_mode,
            $gateway_region,
            $this->getAuthHeaders()
        ));

        // customer order response
        return $tokenization_service->createOrder($customer_uuid, $payload->toArray());
    }

    /**
     * Handle Tokenization
     *
     * @param int $customer_id
     * @param string $customer_uuid
     * @throws InvalidRequest
     */
    public function handleTokenization($customer_id, $customer_uuid)
    {
        $api_mode = $this->config->get('payment_sezzle_test_mode') ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $gateway_region = $this->config->get('payment_sezzle_gateway_region');

        // instantiate order service
        $customerService = new CustomerService(new ClientService(
            $api_mode,
            $gateway_region,
            $this->getAuthHeaders()
        ));

        $customer = $customerService->getCustomer($customer_uuid);
        $this->addTokenizationRecord($customer_id, $customer_uuid, $customer->getTokenExpiration());
    }

    /**
     * Building Session Payload for Checkout
     *
     * @param array $order_info
     * @return Session
     */
    private function buildSessionPayload($order_info)
    {
        // session model
        $session_model = new Session();
        return $session_model->setCompleteUrl($this->getUrlObject(self::REDIRECT_COMPLETE))
            ->setCancelUrl($this->getUrlObject(self::REDIRECT_CANCEL))
            ->setOrder($this->getOrderObject($order_info))
            ->setCustomer($this->getCustomerObject($order_info));
    }

    /**
     * Getting URL Object
     *
     * @param string $action
     * @return Session\Url
     */
    private function getUrlObject($action)
    {
        // url model
        $url_model = new Session\Url();
        $url_model->setMethod(Config::HTTP_POST);
        if ($action === self::REDIRECT_COMPLETE) {
            $href = $this->url->link('extension/payment/sezzle/checkoutComplete');
            return $url_model->setHref($href);

        }
        $href = $this->url->link('checkout/checkout');
        return $url_model->setHref($href);
    }

    /**
     * Getting Order Information
     *
     * @param array $order_info
     * @return Session\Order
     */
    private function getOrderObject($order_info)
    {
        // order model
        $order = new Session\Order();
        return $order->setIntent("AUTH")
            ->setDescription("Opencart Order")
            ->setReferenceId((string)$order_info['order_id'])
            ->setRequiresShippingInfo(false)
            ->setOrderAmount($this->getOrderAmountObject($order_info))
            ->setTaxAmount($this->getTaxAmountObject($order_info))
            ->setShippingAmount($this->getShippingAmountObject($order_info))
            ->setItems($this->getItemsObject($order_info));
    }

    /**
     * Get Order Amount Object
     *
     * @param array $order_info
     * @return Session\Order\Amount
     */
    private function getOrderAmountObject($order_info)
    {
        $amount_cents = Util::formatToCents($order_info['total']);
        return $this->getAmountObject($amount_cents, $order_info['currency_code']);
    }

    /**
     * Get Amount Object
     *
     * @param int $amount_cents
     * @param string $currency_code
     * @return Sezzle\Model\Session\Order\Amount
     */
    private function getAmountObject($amount_cents, $currency_code)
    {
        // amount model
        $amount_model = new Session\Order\Amount();
        return $amount_model->setAmountInCents($amount_cents)->setCurrency($currency_code);
    }

    /**
     * Get Tax Amount Object
     *
     * @param array $order_info
     * @return Session\Order\Amount
     */
    private function getTaxAmountObject($order_info)
    {
        $totals = $this->model_checkout_order->getOrderTotals($order_info['order_id']);
        $tax_amount = 0;
        foreach ($totals as $t) {
            if ($t['code'] == 'tax') {
                if (is_numeric($t['value'])) {
                    $tax_amount += $t['value'];
                }
            }
        }

        $amount_cents = Util::formatToCents($tax_amount);
        return $this->getAmountObject($amount_cents, $order_info['currency_code']);
    }

    /**
     * Get Shipping Amount Object
     *
     * @return Sezzle\Model\Session\Order\Amount
     */
    private function getShippingAmountObject($order_info)
    {
        $totals = $this->model_checkout_order->getOrderTotals($order_info['order_id']);
        $shipping_amount = 0;
        foreach ($totals as $t) {
            if ($t['code'] == 'shipping') {
                if (is_numeric($t['value'])) {
                    $shipping_amount += $t['value'];
                }
            }
        }

        $amount_cents = Util::formatToCents($shipping_amount);
        return $this->getAmountObject($amount_cents, $order_info['currency_code']);
    }

    /**
     * Getting Product Information
     *
     * @param array $order_info
     * @return array
     */
    private function getItemsObject($order_info)
    {
        $this->load->model('catalog/product');

        $products = $this->cart->getProducts($order_info['order_id']);

        $items = [];
        foreach ($products as $key => $product) {
            // item model
            $item = new Session\Order\Item();
            $item->setName($product['name'])
                ->setQuantity((int)$product['quantity'])
                ->setSku($product['model'])
                ->setPrice($this->getItemAmountObject($product['price'], $order_info));
            $items[$key] = $item->toArray();
        }
        return $items;
    }

    /**
     * Get Item Amount Object
     *
     * @param float $amount
     * @param array $order_info
     * @return Session\Order\Amount
     */
    private function getItemAmountObject($amount, $order_info)
    {
        $amount_cents = Util::formatToCents($amount);
        return $this->getAmountObject($amount_cents, $order_info['currency_code']);
    }

    /**
     * Getting Customer Information
     *
     * @param array $order_info
     * @return Session\Customer
     */
    private function getCustomerObject($order_info)
    {
        // customer model
        $customer_model = new Session\Customer();
        $isTokenizeEnabled = $this->config->get('payment_sezzle_tokenization') ? true : false;
        return $customer_model->setEmail($order_info['email'])
            ->setFirstName($order_info['firstname'])
            ->setLastName($order_info['lastname'])
            ->setPhone($order_info['telephone'])
            ->setTokenize($isTokenizeEnabled)
            ->setBillingAddress($this->getAddressObject("billing", $order_info))
            ->setShippingAddress($this->getAddressObject("shipping", $order_info));
    }

    /**
     * Getting Shipping and Billing Address
     *
     * @param string $type
     * @param array $order_info
     * @return Session\Customer\Address
     */
    private function getAddressObject($type, $order_info)
    {
        // address model
        $address_model = new Session\Customer\Address();

        if ($type === "billing") {
            return $address_model->setName(sprintf('%s %s', $order_info['payment_firstname'], $order_info['payment_lastname']))
                ->setStreet($order_info['payment_address_1'])
                ->setStreet2($order_info['payment_address_2'])
                ->setState($order_info['payment_zone'])
                ->setCity($order_info['payment_city'])
                ->setCountryCode($order_info['payment_country'])
                ->setPhoneNumber($order_info['telephone'])
                ->setPostalCode($order_info['payment_postcode']);
        }

        return $address_model->setName(sprintf('%s %s', $order_info['shipping_firstname'], $order_info['shipping_lastname']))
            ->setStreet($order_info['shipping_address_1'])
            ->setStreet2($order_info['shipping_address_2'])
            ->setState($order_info['shipping_zone'])
            ->setCity($order_info['shipping_city'])
            ->setCountryCode($order_info['shipping_country'])
            ->setPhoneNumber($order_info['telephone'])
            ->setPostalCode($order_info['shipping_postcode']);

    }

    /**
     * Build Capture Payload
     *
     * @param array $order_info
     * @return Sezzle\Model\Order\Capture
     */
    public function buildCapturePayload($order_info)
    {
        $capture_model = new Sezzle\Model\Order\Capture();
        return $capture_model->setCaptureAmount(
            $this->getAmountObject(
                Util::formatToCents($order_info['total']),
                $order_info['currency_code']
            )
        )
            ->setPartialCapture(false);
    }

    /**
     * Building Customer Payload for Checkout
     *
     * @param array $order_info
     * @return CustomerOrder
     */
    private function buildCustomerOrderPayload($order_info)
    {
        $payload = new CustomerOrder();
        return $payload->setIntent('AUTH')
            ->setReferenceId((string)$order_info['order_id'])
            ->setOrderAmount($this->getOrderAmountObject($order_info));
    }

    /****************************
     * SEZZLE SERVICE FUNCTIONS *
     ****************************/

    /********************
     * HELPER FUNCTIONS *
     ********************/

    /**
     * Setting Authorization
     *
     * @param Order $sezzle_order
     * @param array $order_info
     * @return bool
     */
    public function setAuthorization($sezzle_order, $order_info)
    {
        $authorization = $sezzle_order->getAuthorization();
        if (!$authorization || !$authorization->isApproved()) {
            return false;
        }

        // validating amount
        $amount = $authorization->getAuthorizationAmount()->getAmountInCents();
        if ($amount <= 0) {
            return false;
        }

        // setting auth amount
        $authorized_amount = (float)$amount / 100;
        $this->setAuthorizedAmount($order_info, $authorized_amount);

        // set auth expiration if txn method id Authorize Only
        if ($this->config->get('payment_sezzle_transaction_method') === self::MODE_AUTH) {
            $this->setAuthExpiration($authorization->getExpiration(), $sezzle_order->getUuid());
        }

        return true;
    }

    /**
     * Checking if the Order Total matched with Authorized Amount
     *
     * @param array $order_info
     * @param Order $sezzle_order
     * @return bool
     */
    public function isAmountMatched($order_info, $sezzle_order)
    {
        $authorized_amount_in_cents = $sezzle_order->getAuthorization()->getAuthorizationAmount()->getAmountInCents();
        if ($authorized_amount_in_cents <= 0) {
            return false;
        }

        $authorized_amount = $authorized_amount_in_cents / 100;
        return round($order_info['total'], 2) === (float)$authorized_amount;
    }

    /**
     * Getting order status id by order state
     *
     * @param string $state
     * @return bool
     */
    public function getOrderStatusId($state)
    {
        $this->load->model('localisation/order_status');

        $order_statuses = $this->model_localisation_order_status->getOrderStatuses();
        foreach ($order_statuses as $order_status) {
            switch ($state) {
                case self::ORDER_STATE_CAPTURED:
                    if ($order_status['name'] === 'Processing') {
                        return $order_status['order_status_id'];
                    }
                    break;
                case self::ORDER_STATE_REFUNDED:
                    if ($order_status['name'] === 'Refunded') {
                        return $order_status['order_status_id'];
                    }
                    break;
                case self::ORDER_STATE_RELEASED:
                    if ($order_status['name'] === 'Voided') {
                        return $order_status['order_status_id'];
                    }
                    break;
            }
        }
        return false;
    }

    /**
     * Get Customer UUID
     *
     * @param int $customer_id
     * @return false|array
     * @throws Exception
     */
    public function getCustomerUUID($customer_id)
    {
        $date_now = new DateTime();
        $tokenization_details = $this->getTokenizationRecord($customer_id);
        if (empty($tokenization_details)) {
            return false;
        }

        if (new DateTime($tokenization_details['customer_uuid_expiration']) < $date_now || !$tokenization_details['approved']) {
            $this->deleteTokenizationRecord($customer_id);
            return false;
        }
        return $tokenization_details['customer_uuid'];
    }

    /********************
     * HELPER FUNCTIONS *
     ********************/

    /****************
     * DB FUNCTIONS *
     ****************/

    /**
     * Get Sezzle order by OC order ID
     *
     * @param int $order_id
     * @return mixed
     */
    public function getOrder($order_id)
    {
        $query = $this->db->query(sprintf("SELECT * FROM %s WHERE opencart_order_id = %d LIMIT 1", DB_PREFIX . "sezzle_order", (int)$order_id));
        if (!$query->num_rows) {
            return false;
        }

        $query->row['transactions'] = $this->getTransactions($query->row['sezzle_order_id']);
        return $query->row;
    }

    /**
     * Add Sezzle order transaction
     *
     * @param int $sezzle_order_id
     * @param float $amount
     * @param string $type
     */
    public function addTransaction($sezzle_order_id, $amount, $type)
    {
        $this->db->query("INSERT INTO `" . DB_PREFIX . "sezzle_order_transaction` SET `sezzle_order_id` = " . (int)$sezzle_order_id . ", `date_added` = now(), `type` = '" . $this->db->escape($type) . "', `amount` = " . (float)$amount);
    }

    /**
     * Add Tokenization Record
     *
     * @param int $customer_id
     * @param string $customer_uuid
     * @param string $customer_uuid_expiration
     */
    private function addTokenizationRecord($customer_id, $customer_uuid, $customer_uuid_expiration)
    {
        $this->db->query("INSERT INTO `" . DB_PREFIX . "sezzle_tokenization` SET `customer_id` = " . (int)$customer_id . ", `customer_uuid` = '" . $this->db->escape($customer_uuid) . "', `customer_uuid_expiration` = '" . $customer_uuid_expiration . "', `approved` = true");
    }

    /**
     * Getting Tokenization Record
     *
     * @param int $customer_id
     * @return array
     */
    public function getTokenizationRecord($customer_id)
    {
        $query = $this->db->query(sprintf("SELECT * FROM %s WHERE customer_id = %d LIMIT 1", DB_PREFIX . "sezzle_tokenization", (int)$customer_id));
        return $query->row;
    }

    /**
     * Deleting Tokenization Record
     *
     * @param int $customer_id
     */
    public function deleteTokenizationRecord($customer_id)
    {
        $this->db->query("DELETE FROM `" . DB_PREFIX . "sezzle_tokenization` WHERE `customer_id` = " . (int)$customer_id);
    }

    /**
     * Adding Order Info to DB
     *
     * @param array $order_info
     * @param array $sezzleCheckoutData
     */
    public function addOrder($order_info, $sezzleCheckoutData)
    {
        $this->db->query("INSERT INTO `" . DB_PREFIX . "sezzle_order` SET `date_added` = now(), `date_modified` = now(), `opencart_order_id` = " . (int)$order_info['order_id'] . ", `order_uuid` = '" . $this->db->escape($sezzleCheckoutData['order_uuid']) . "', `checkout_url` = '" . $this->db->escape($sezzleCheckoutData['checkout_url']) . "', `checkout_amount` = " . (float)$order_info['total'] . ", `currency_code` = '" . $this->db->escape($order_info['currency_code']) . "', `currency_value` = " . (float)$order_info['currency_value'] . ", `transaction_method` = '" . $this->db->escape($order_info['transaction_method']) . "'");
    }

    /**
     * Setting Captured Amount
     *
     * @param int $order_id
     * @param float $amount
     */
    public function setCapturedAmount($order_id, $amount)
    {
        $this->db->query("UPDATE " . DB_PREFIX . "sezzle_order SET `captured_amount` = " . (float)$amount . " WHERE opencart_order_id = " . (int)$order_id);
    }

    /**
     * Setting Authorized Amount
     *
     * @param array $order_info
     * @param float $amount
     */
    private function setAuthorizedAmount($order_info, $amount)
    {
        $this->db->query("UPDATE " . DB_PREFIX . "sezzle_order SET `authorized_amount` = " . (float)$amount . " WHERE opencart_order_id = " . (int)$order_info['order_id']);
    }

    /**
     * Setting Auth Expiration
     *
     * @param DateTime $auth_expiration
     * @param string $order_uuid
     */
    private function setAuthExpiration($auth_expiration, $order_uuid)
    {
        $this->db->query("UPDATE " . DB_PREFIX . "sezzle_order SET `auth_expiration` ='" . $auth_expiration . "'  WHERE order_uuid = '" . $order_uuid . "'");
    }

    /**
     * Get Sezzle order transaction
     *
     * @param int $sezzle_order_id
     * @return array
     */
    private function getTransactions($sezzle_order_id)
    {
        return $this->db->query(sprintf("SELECT * FROM %s WHERE sezzle_order_id = %d", DB_PREFIX . "sezzle_order_transaction", (int)$sezzle_order_id))->rows;
    }

    /****************
     * DB FUNCTIONS *
     ****************/
}


