<?php

namespace Sezzle\HttpClient;

use Exception;
use Sezzle\Config;
use Sezzle\HttpClient\Exception\InvalidRequest;

class ClientService
{

    /**
     * @var string
     */
    private $gatewayUrl;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var string
     */
    private $apiMode;

    /**
     * @var string
     */
    private $gatewayRegion;

    /**
     * ClientService constructor.
     * @param string $apiMode
     * @param string $gatewayRegion
     * @param array $headers
     */
    public function __construct(
        string $apiMode,
        string $gatewayRegion = "",
        array $headers = []
    )
    {
        $this->apiMode = $apiMode;
        $this->gatewayRegion = $gatewayRegion;
        $this->gatewayUrl = Config::getGatewayUrl($this->apiMode, $this->gatewayRegion);

        $this->client = new Client($this->gatewayUrl, $headers, Config::API_V2);
    }

    /**
     * Sends a request and returns the response.
     * The type can be obtained from Config.php
     *
     * @param string $type
     * @param string $route
     * @param array $data
     * @return array
     * @throws InvalidRequest
     */
    public function sendRequest($type, $route, array $data = [])
    {
        switch ($type) {
            case Config::HTTP_POST:
                $response = $this->client->_($route)->post($data);
                break;

            case Config::HTTP_GET:
                $response = $this->client->_($route)->get();
                break;

            case Config::HTTP_PATCH:
                $response = $this->client->_($route)->patch($data);
                break;

            case Config::HTTP_PUT:
                $response = $this->client->_($route)->put($data);
                break;

            default:
                throw new InvalidRequest('An unsupported request type was provided. The type was: .' . $type);
        }

        // validating the service call
        $statusCode = $response->statusCode();
        if (!$this->validStatusCode($type, $statusCode)) {
            throw new Exception(sprintf('Error occurred while processing the request. Route : %s | Status Code : %d', $route, $statusCode));
        }

        return [
            "body" => json_decode($response->body(), true),
            "status_code" => $statusCode
        ];
    }

    /**
     * Validating API response status code
     *
     * @param string $type
     * @param int $code
     * @return bool
     */
    private function validStatusCode($type, $code)
    {
        switch ($type) {
            case Config::HTTP_POST:
                return $code === 201 || $code === 200;

            case Config::HTTP_PUT:
            case Config::HTTP_GET:
                return $code === 200;

            case Config::HTTP_PATCH:
                return $code === 204;

            default:
                return false;
        }
    }
}
