<?php
require_once DIR_SYSTEM . 'library/vendor/sezzle/php-sdk/autoload.php';

use Sezzle\Config;
use Sezzle\HttpClient\ClientService;
use Sezzle\HttpClient\Exception\InvalidRequest;
use Sezzle\Model\AuthCredentials;
use Sezzle\Services\AuthenticationService;

/**
 * Class ControllerExtensionPaymentSezzle
 */
class ControllerExtensionPaymentSezzle extends Controller
{
    const MODE_SANDBOX = "sandbox";
    const MODE_PRODUCTION = "production";
    const SUPPORTED_REGIONS = ['US', 'EU'];
    /**
     * @var array
     */
    private $error = array();

    /**
     * Configuration for Sezzle
     */
    public function index()
    {
        $data = $this->load->language('extension/payment/sezzle');
        $this->document->setTitle($this->language->get('heading_title'));
        $this->load->model('setting/setting');

        if (($this->request->server['REQUEST_METHOD'] === Config::HTTP_POST)) {
            $regionData = $this->getGatewayRegion();
            if (empty($regionData)) {
                $this->error['error_config'] = $this->language->get('error_credentials');
            } else {
                $post_data = $this->request->post;
                $post_data = array_merge($post_data, $regionData);

                $this->model_setting_setting->editSetting('payment_sezzle', $post_data);

                try {
                    // sending config data to Sezzle
                    $this->load->model('extension/payment/sezzle');
                    $this->model_extension_payment_sezzle->sendConfig($post_data);
                } catch (Exception $e) {
                    // ignore error and continue saving the config
                }

                $this->session->data['success'] = $this->language->get('text_success');
                if (isset($this->request->get['continue']) && $this->request->get['continue']) {
                    $this->response->redirect($this->url->link('extension/payment/sezzle', 'user_token=' . $this->session->data['user_token'], true));
                } else {
                    $this->response->redirect($this->url->link('extension/extension', 'user_token=' . $this->session->data['user_token'] . '&type=payment', true));
                }
            }
        }

        $data['user_token'] = $this->session->data['user_token'];

        if (isset($this->session->data['success'])) {
            $data['success'] = $this->session->data['success'];

            unset($this->session->data['success']);
        } else {
            $data['success'] = '';
        }

        if (isset($this->error['error_config'])) {
            $data['error_config'] = $this->error['error_config'];
        } else {
            $data['error_config'] = '';
        }

        if (isset($this->error['public_key'])) {
            $data['error_public_key'] = $this->error['public_key'];
        } else {
            $data['error_public_key'] = '';
        }

        if (isset($this->error['private_key'])) {
            $data['error_private_key'] = $this->error['private_key'];
        } else {
            $data['error_private_key'] = '';
        }

        $data['breadcrumbs'] = array();
        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('text_home'),
            'href' => $this->url->link('common/dashboard', 'user_token=' . $this->session->data['user_token'], true),
        );

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('text_extension'),
            'href' => $this->url->link('marketplace/extension', 'user_token=' . $this->session->data['user_token'] . '&type=payment', true),
        );

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('heading_title'),
            'href' => $this->url->link('extension/payment/sezzle', 'user_token=' . $this->session->data['user_token'], true),
        );

        $data['action'] = $this->url->link('extension/payment/sezzle', 'user_token=' . $this->session->data['user_token'], true);
        $data['continue'] = $this->url->link('extension/payment/sezzle', 'user_token=' . $this->session->data['user_token'] . '&continue=1', true);
        $data['cancel'] = $this->url->link('marketplace/extension', 'user_token=' . $this->session->data['user_token'] . '&type=payment', true);

        if (isset($this->request->post['payment_sezzle_status'])) {
            $data['payment_sezzle_status'] = $this->request->post['payment_sezzle_status'];
        } else {
            $data['payment_sezzle_status'] = $this->config->get('payment_sezzle_status');
        }

        if (isset($this->request->post['payment_sezzle_public_key'])) {
            $data['payment_sezzle_public_key'] = $this->request->post['payment_sezzle_public_key'];
        } else {
            $data['payment_sezzle_public_key'] = $this->config->get('payment_sezzle_public_key');
        }

        if (isset($this->request->post['payment_sezzle_private_key'])) {
            $data['payment_sezzle_private_key'] = $this->request->post['payment_sezzle_private_key'];
        } else {
            $data['payment_sezzle_private_key'] = $this->config->get('payment_sezzle_private_key');
        }

        if (isset($this->request->post['payment_sezzle_test_mode'])) {
            $data['payment_sezzle_test_mode'] = $this->request->post['payment_sezzle_test_mode'];
        } else {
            $data['payment_sezzle_test_mode'] = $this->config->get('payment_sezzle_test_mode');
        }

        if (isset($this->request->post['payment_sezzle_transaction_method'])) {
            $data['payment_sezzle_transaction_method'] = $this->request->post['payment_sezzle_transaction_method'];
        } else {
            $data['payment_sezzle_transaction_method'] = $this->config->get('payment_sezzle_transaction_method');
        }

        if (isset($this->request->post['payment_sezzle_tokenization'])) {
            $data['payment_sezzle_tokenization'] = $this->request->post['payment_sezzle_tokenization'];
        } else {
            $data['payment_sezzle_tokenization'] = $this->config->get('payment_sezzle_tokenization');
        }

        if (isset($this->request->post['payment_sezzle_widget_pdp'])) {
            $data['payment_sezzle_widget_pdp'] = $this->request->post['payment_sezzle_widget_pdp'];
        } else {
            $data['payment_sezzle_widget_pdp'] = $this->config->get('payment_sezzle_widget_pdp');
        }

        if (isset($this->request->post['payment_sezzle_widget_cart'])) {
            $data['payment_sezzle_widget_cart'] = $this->request->post['payment_sezzle_widget_cart'];
        } else {
            $data['payment_sezzle_widget_cart'] = $this->config->get('payment_sezzle_widget_cart');
        }

        $data['header'] = $this->load->controller('common/header');
        $data['column_left'] = $this->load->controller('common/column_left');
        $data['footer'] = $this->load->controller('common/footer');

        $this->response->setOutput($this->load->view('extension/payment/sezzle', $data));
    }

    /**
     * Getting Gateway Region
     *
     * @return array
     */
    public function getGatewayRegion()
    {
        $stored_gateway_region = $this->config->get('payment_sezzle_gateway_region');
        if (!$this->hasKeysConfigurationChanged() && $stored_gateway_region) {
            return [
                "payment_sezzle_merchant_uuid" => $this->config->get('payment_sezzle_merchant_uuid'),
                "payment_sezzle_gateway_region" => $stored_gateway_region
            ];
        }

        foreach (self::SUPPORTED_REGIONS as $region) {
            try {
                if ($merchant_uuid = $this->validateAPIKeys($region)) {
                    return [
                        "payment_sezzle_merchant_uuid" => $merchant_uuid,
                        "payment_sezzle_gateway_region" => $region
                    ];
                }
            } catch (Exception $e) {
                continue;
            }
        }
        return [];
    }

    /**
     * Checking if API Keys Configuration has changed or not
     *
     * @return bool
     */
    private function hasKeysConfigurationChanged()
    {
        $stored_api_mode = $this->config->get('payment_sezzle_test_mode') ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $stored_public_key = $this->config->get('payment_sezzle_public_key');
        $stored_private_key = $this->config->get('payment_sezzle_private_key');

        $api_mode = $this->request->post['payment_sezzle_test_mode'] ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $public_key = $this->request->post['payment_sezzle_public_key'];
        $private_key = $this->request->post['payment_sezzle_private_key'];

        return !($stored_public_key === $public_key
            && $stored_private_key === $private_key
            && $stored_api_mode === $api_mode);
    }

    /**
     * Validating API keys and returning Merchant UUID
     *
     * @param string $region
     * @return string
     * @throws InvalidRequest
     */
    private function validateAPIKeys($region = "")
    {
        $api_mode = $this->request->post['payment_sezzle_test_mode'] ? self::MODE_SANDBOX : self::MODE_PRODUCTION;
        $public_key = $this->request->post['payment_sezzle_public_key'];
        $private_key = $this->request->post['payment_sezzle_private_key'];

        // auth credentials set
        $auth_model = new AuthCredentials();
        $auth_model->setPublicKey($public_key)->setPrivateKey($private_key);

        // instantiate authentication service
        $token_service = new AuthenticationService(new ClientService(
            $api_mode,
            $region
        ));

        // get token
        $token_model = $token_service->get($auth_model->toArray());
        return $token_model->getMerchantUUID();
    }

    /**
     * Install event
     */
    public function install()
    {
        if (!$this->user->hasPermission('modify', 'extension/extension/payment')) {
            return;
        }

        $this->load->model('extension/payment/sezzle');

        $this->model_extension_payment_sezzle->install();

        $this->load->model('setting/event');

        $this->model_setting_event->addEvent('payment_sezzle', 'catalog/view/checkout/cart/after', 'extension/payment/sezzle/eventPostViewCheckoutCart');
        $this->model_setting_event->addEvent('payment_sezzle', 'catalog/view/product/product/after', 'extension/payment/sezzle/eventPostViewProductProduct');
    }

    /**
     * Uninstall event
     */
    public function uninstall()
    {
        if (!$this->user->hasPermission('modify', 'extension/extension/payment')) {
            return;
        }

        $this->load->model('extension/payment/sezzle');

        $this->model_extension_payment_sezzle->uninstall();

        $this->load->model('setting/event');

        $this->model_setting_event->deleteEventByCode('payment_sezzle');
    }

    /**
     * Sezzle section in Order page admin
     *
     * @return mixed
     * @throws Exception
     */
    public function order()
    {
        if (!$this->config->get('payment_sezzle_status') || !isset($this->request->get['order_id'])) {
            return;
        }
        $this->load->model('extension/payment/sezzle');

        $sezzle_order = $this->model_extension_payment_sezzle->getOrder($this->request->get['order_id']);
        if (empty($sezzle_order)) {
            return;
        }

        $data = $this->load->language('extension/payment/sezzle');

        $dateTimeNow = new DateTime();
        $dateTimeAuthExpire = new DateTime($sezzle_order['auth_expiration']);
        $auth_expired = $dateTimeNow > $dateTimeAuthExpire;

        $actual_authorized_amount = $sezzle_order['authorized_amount'] - $sezzle_order['released_amount'];

        // determining capture, refund and release availability
        $can_refund = $can_capture = $can_release = false;
        if (!$auth_expired && ($actual_authorized_amount > $sezzle_order['captured_amount'])) {
            $can_capture = $can_release = true;
        }

        if ($sezzle_order['captured_amount'] > $sezzle_order['refunded_amount']) {
            $can_refund = true;
        }

        // sezzle payment data
        $sezzle_order['authorized_amount'] = $this->currency->format($sezzle_order['authorized_amount'], $sezzle_order['currency_code'], $sezzle_order['currency_value']);
        $sezzle_order['captured_amount'] = $this->currency->format($sezzle_order['captured_amount'], $sezzle_order['currency_code'], $sezzle_order['currency_value']);
        $sezzle_order['refunded_amount'] = $this->currency->format($sezzle_order['refunded_amount'], $sezzle_order['currency_code'], $sezzle_order['currency_value']);
        $sezzle_order['released_amount'] = $this->currency->format($sezzle_order['released_amount'], $sezzle_order['currency_code'], $sezzle_order['currency_value']);
        $sezzle_order['auth_expired'] = $dateTimeNow > $dateTimeAuthExpire;
        $sezzle_order['transaction_method'] = $sezzle_order['transaction_method'] === ModelExtensionPaymentSezzle::MODE_AUTH ?
            $this->language->get('text_auth') : $this->language->get('text_auth_capture');

        // adding txn data data
        foreach ($sezzle_order['transactions'] as $key => $txn) {
            $sezzle_order['transactions'][$key]['amount'] = $this->currency->format($txn['amount'], $sezzle_order['currency_code'], $sezzle_order['currency_value']);
        }

        // template params
        $template_params = array_merge($data, [
            'sezzle_order' => $sezzle_order,
            'order_id' => $this->request->get['order_id'],
            'user_token' => $this->session->data['user_token'],
            'currency_symbol' => $this->currency->getSymbolLeft($sezzle_order['currency_code']),
            'can_capture' => $can_capture,
            'can_refund' => $can_refund,
            'can_release' => $can_release
        ]);

        return $this->load->view('extension/payment/sezzle_order', $template_params);
    }

    /**
     * Capturing the Payment
     */
    public function capture()
    {
        $response = array();

        // loading models and language
        $this->load->language('extension/payment/sezzle');
        $this->load->model('extension/payment/sezzle');

        $order_id = $this->request->get['order_id'];

        // getting sezzle order record
        $txn = $this->model_extension_payment_sezzle->getOrder($order_id);
        if (!$txn) {
            $response['error'] = $this->language->get('error_sezzle_txn_not_found');
            $this->renderResponse($response);
            return;
        }

        // validating amount
        $amount = (float)$this->request->post['amount'];
        if (!$this->model_extension_payment_sezzle->validateAmount($txn, $amount, ModelExtensionPaymentSezzle::PAYMENT_ACTION_CAPTURE)) {
            $response['error'] = $this->language->get('error_invalid_amount');
            $this->renderResponse($response);
            return;
        }

        // getting order status id by order state
        $order_status_id = $this->model_extension_payment_sezzle->getOrderStatusId(ModelExtensionPaymentSezzle::ORDER_STATE_APPROVED);
        if (!$order_status_id) {
            $response['error'] = $this->language->get('error_order_status');
            $this->renderResponse($response);
            return;
        }

        $is_partial_capture = (float)$amount !== (float)$txn['authorized_amount'];

        // capturing payment
        try {
            $this->model_extension_payment_sezzle->capturePayment($amount, $txn['currency_code'], $txn['order_uuid'], $is_partial_capture);
            $final_captured_amount = $amount + $txn['captured_amount'];
            $this->model_extension_payment_sezzle->setCapturedAmount($order_id, $final_captured_amount);
            $this->model_extension_payment_sezzle->addTransaction($txn['sezzle_order_id'], $amount, ModelExtensionPaymentSezzle::PAYMENT_ACTION_CAPTURE);
            $this->model_extension_payment_sezzle->addOrderHistory($order_id, $order_status_id);
            $response = [
                'final_amount' => $this->currency->format($final_captured_amount, $txn['currency_code'], $txn['currency_value']),
                'success' => sprintf($this->language->get('text_captured'), $order_id)
            ];
        } catch (Exception $e) {
            $response['error'] = $this->language->get('error_capture');
        }

        $this->renderResponse($response);
    }

    /**
     * Refunding the payment
     */
    public function refund()
    {
        $response = array();

        // loading models and language
        $this->load->language('extension/payment/sezzle');
        $this->load->model('extension/payment/sezzle');

        $order_id = $this->request->get['order_id'];

        // getting sezzle order record
        $txn = $this->model_extension_payment_sezzle->getOrder($order_id);
        if (!$txn) {
            $response['error'] = $this->language->get('error_sezzle_txn_not_found');
            $this->renderResponse($response);
            return;
        }

        // validating amount
        $amount = (float)$this->request->post['amount'];
        if (!$this->model_extension_payment_sezzle->validateAmount($txn, $amount, ModelExtensionPaymentSezzle::PAYMENT_ACTION_REFUND)) {
            $response['error'] = $this->language->get('error_invalid_amount');
            $this->renderResponse($response);
            return;
        }

        $final_refunded_amount = $amount + $txn['refunded_amount'];
        $full_refund = (float)$final_refunded_amount === (float)$txn['captured_amount'];

        // getting order status id by order state
        $order_status_id = false;
        if ($full_refund) {
            $order_status_id = $this->model_extension_payment_sezzle->getOrderStatusId(ModelExtensionPaymentSezzle::ORDER_STATE_REFUNDED);
            if (!$order_status_id) {
                $response['error'] = $this->language->get('error_order_status');
                $this->renderResponse($response);
                return;
            }
        }

        // refunding payment
        try {
            $this->model_extension_payment_sezzle->refundPayment($txn['order_uuid'], $amount, $txn['currency_code']);
            $this->model_extension_payment_sezzle->setRefundedAmount($order_id, $final_refunded_amount);
            $this->model_extension_payment_sezzle->addTransaction($txn['sezzle_order_id'], $amount, ModelExtensionPaymentSezzle::PAYMENT_ACTION_REFUND);
            if ($order_status_id) {
                $this->model_extension_payment_sezzle->addOrderHistory($order_id, $order_status_id);
            }
            $response = [
                'final_amount' => $this->currency->format($final_refunded_amount, $txn['currency_code'], $txn['currency_value']),
                'success' => sprintf($this->language->get('text_refunded'), $order_id)
            ];
        } catch (Exception $e) {
            $response['error'] = $this->language->get('error_refund');
        }

        $this->renderResponse($response);
    }

    /**
     * Releasing the payment
     */
    public function release()
    {
        $response = array();

        // loading models and language
        $this->load->language('extension/payment/sezzle');
        $this->load->model('extension/payment/sezzle');

        $order_id = $this->request->get['order_id'];

        // getting sezzle order record
        $txn = $this->model_extension_payment_sezzle->getOrder($order_id);
        if (!$txn) {
            $response['error'] = $this->language->get('error_sezzle_txn_not_found');
            $this->renderResponse($response);
            return;
        }

        // validating amount
        $amount = (float)$this->request->post['amount'];
        if (!$this->model_extension_payment_sezzle->validateAmount($txn, $amount, ModelExtensionPaymentSezzle::PAYMENT_ACTION_RELEASE)) {
            $response['error'] = $this->language->get('error_invalid_amount');
            $this->renderResponse($response);
            return;
        }

        $final_released_amount = $amount + $txn['released_amount'];
        $full_release = (float)$txn['captured_amount'] === 0.00 && (float)$final_released_amount === (float)$txn['authorized_amount'];

        // getting order status id by order state
        $order_status_id = false;
        if ($full_release) {
            $order_status_id = $this->model_extension_payment_sezzle->getOrderStatusId(ModelExtensionPaymentSezzle::ORDER_STATE_RELEASED);
            if (!$order_status_id) {
                $response['error'] = $this->language->get('error_order_status');
                $this->renderResponse($response);
                return;
            }
        }

        // releasing payment
        try {
            $this->model_extension_payment_sezzle->releasePayment($txn['order_uuid'], $amount, $txn['currency_code']);
            $final_released_amount = $amount + $txn['released_amount'];
            $this->model_extension_payment_sezzle->setReleasedAmount($order_id, $final_released_amount);
            $this->model_extension_payment_sezzle->addTransaction($txn['sezzle_order_id'], $amount, ModelExtensionPaymentSezzle::PAYMENT_ACTION_RELEASE);
            if ($order_status_id) {
                $this->model_extension_payment_sezzle->addOrderHistory($order_id, $order_status_id);
            }
            $response = [
                'final_amount' => $this->currency->format($final_released_amount, $txn['currency_code'], $txn['currency_value']),
                'success' => sprintf($this->language->get('text_released'), $order_id)
            ];
        } catch (Exception $e) {
            $response['error'] = $this->language->get('error_release');
        }

        $this->renderResponse($response);
    }

    /**
     * Render JSON response
     *
     * @param array $data
     */
    private function renderResponse($data)
    {
        $this->response->addHeader('Content-Type: application/json');
        $this->response->setOutput(json_encode($data));
    }

}
